/*******************************************************************************
Copyright (c) 2014 released Microchip Technology Inc.  All rights reserved.

Microchip licenses to you the right to use, modify, copy and distribute
Software only when embedded on a Microchip microcontroller or digital signal
controller that is integrated into your product or third party product
(pursuant to the sublicense terms in the accompanying license agreement).

You should refer to the license agreement accompanying this Software for
additional information regarding your rights and obligations.

SOFTWARE AND DOCUMENTATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
MERCHANTABILITY, TITLE, NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
IN NO EVENT SHALL MICROCHIP OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER
CONTRACT, NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR
OTHER LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE OR
CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT OF
SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
(INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.
*******************************************************************************/


#include "mcp2515.h"
#include "define.h"

#include <spi.h>

void DRV_MCP2515_Reset()
{
    CAN_CS_LOW();
    SPITransfer(CAN_RESET);
    CAN_CS_HIGH();

    long i = 1000;
    while (i--);
}

DRV_MCP2515_OPERATION_MODE DRV_MCP2515_OperationModeGet()
{
    DRV_MCP2515_REG_CANSTAT canStat;

    Nop();
    canStat.byte = DRV_MCP2515_RegisterRead(cREGADDR_CANSTAT);

    return canStat.OpMode;
}

void DRV_MCP2515_OperationModeSelect(DRV_MCP2515_OPERATION_MODE opMode)
{
    DRV_MCP2515_REG_CANCTRL canCtrl;

    // Read
    Nop();
    canCtrl.byte = DRV_MCP2515_RegisterRead(cREGADDR_CANCTRL);

    //Modify
    canCtrl.OpModeRequest = opMode;

    // Write
    DRV_MCP2515_RegisterWrite(cREGADDR_CANCTRL, canCtrl.byte);

    // Wait till mode has changed
    DRV_MCP2515_REG_CANSTAT canStat;
    canStat.byte = DRV_MCP2515_RegisterRead(cREGADDR_CANSTAT);

    while (canStat.OpMode != canCtrl.OpModeRequest) {
        canStat.byte = DRV_MCP2515_RegisterRead(cREGADDR_CANSTAT);
        Nop();
        Nop();
    }
}

void DRV_MCP2515_ConfigurationSet(DRV_MCP2515_BIT_TIME_CONFIG *config)
{
    DRV_MCP2515_REG_CNF1 cnf1;
    DRV_MCP2515_REG_CNF2 cnf2;
    DRV_MCP2515_REG_CNF3 cnf3;

    cnf1.BRP = config->BRP;
    cnf1.SJW = config->SJW;

    cnf2.PropSeg = config->PropSeg;
    cnf2.PhaseSeg1 = config->PhaseSeg1;
    cnf2.SAM = config->BusSampling;
    cnf2.BTLMode = config->PhaseSeg2Config;

    cnf3.PhaseSeg2 = config->PhaseSeg2;
    cnf3.WakeUpFilterEnable = config->WakeUpFilterEnable;
    cnf3.SOF = config->SOFPinConfig;

    DRV_MCP2515_RegisterWrite(cREGADDR_CNF1, cnf1.byte);
    DRV_MCP2515_RegisterWrite(cREGADDR_CNF2, cnf2.byte);
    DRV_MCP2515_RegisterWrite(cREGADDR_CNF3, cnf3.byte);
}

void DRV_MCP2515_TransmitChannelLoad(DRV_MCP2515_TX_CHANNEL channel, DRV_MCP2515_TX_MSGOBJ* txObj, const unsigned char *txd, unsigned char nBytes)
{
#define TX_BUFFER_SIZE 13

    unsigned char d[TX_BUFFER_SIZE];
    uint32_t reg = 0;
    unsigned char n=0;

    // TXBnSIDH
    Nop(); Nop();
    reg = txObj->id.SID>>3;
    d[n] = reg & 0xff;
    n++;

    // TXBnSIDL
    DRV_MCP2515_REG_TXBnSIDL sidl;
    sidl.byte = 0;
    sidl.SIDL = txObj->id.SID & 0x7;
    sidl.IDE = txObj->ctrl.IDE;
    reg = (txObj->id.EID>>16);
    sidl.EIDHH = reg & 0x3;
    d[n] = sidl.byte;
    n++;

    // TXBnEID8
    reg = (txObj->id.EID>>8);
    d[n] = reg & 0xff;
    n++;

    // TXBnEID0
    d[n] = txObj->id.EID & 0xff;
    n++;

    // TXBnDLC
    DRV_MCP2515_REG_TXBnDLC dlc;
    dlc.byte = 0;
    dlc.DLC = txObj->ctrl.DLC;
    dlc.RTR = txObj->ctrl.RTR;
    d[n] = dlc.byte;
    n++;

    // Data Bytes
    unsigned char i=0;
    unsigned max = nBytes;
    if (max>8) {
        max = 8;
    }
    for(i=0; i<max; i++) {
        d[n] = txd[i];
        n++;
    }

    // Copy data
    unsigned char address;
    switch(channel) {
        case DRV_MCP2515_TX_CH0: address = cREGADDR_TXB0ID; break;
        case DRV_MCP2515_TX_CH1: address = cREGADDR_TXB1ID; break;
        case DRV_MCP2515_TX_CH2: address = cREGADDR_TXB2ID; break;
        default: address = cREGADDR_TXB0ID; break;
    }

    // Write buffer
    DRV_MCP2515_BufferWrite(address, d, n);
}

void DRV_MCP2515_TransmitChannelFlush(DRV_MCP2515_TX_CHANNEL channel)
{
    unsigned char cmd = 0;

    // Get command
    Nop(); Nop();
    
    switch(channel) {
        case DRV_MCP2515_TX_CH0: cmd = CAN_RTS_TXB0; break;
        case DRV_MCP2515_TX_CH1: cmd = CAN_RTS_TXB1; break;
        case DRV_MCP2515_TX_CH2: cmd = CAN_RTS_TXB2; break;
        default: cmd = CAN_RTS_TXB0; break;
    }

    CAN_CS_LOW();
    SPITransfer(cmd);
    CAN_CS_HIGH();
    Nop();
    Nop();
}

void DRV_MCP2515_FilterConfigure(DRV_MCP2515_FILTER filter, DRV_MCP2515_FILTEROBJ_ID* fObj)
{
#define FILTER_SIZE 4

    unsigned char d[FILTER_SIZE];
    uint32_t reg = 0;
    unsigned char n=0;

    // RXFnSIDH
    Nop(); Nop();
    reg = fObj->SID>>3;
    d[n] = reg & 0xff;
    n++;

    // RXFnSIDL
    DRV_MCP2515_REG_TXBnSIDL sidl;
    sidl.byte = 0;
    sidl.SIDL = fObj->SID & 0x7;
    sidl.IDE = fObj->EXIDE;
    reg = (fObj->EID>>16);
    sidl.EIDHH = reg & 0x3;
    d[n] = sidl.byte;
    n++;

    // RXFnEID8
    reg = (fObj->EID>>8);
    d[n] = reg & 0xff;
    n++;

    // RXFnEID0
    d[n] = fObj->EID & 0xff;
    n++;

        // Copy data
    unsigned char address;
    switch(filter) {
        case DRV_MCP2515_FILTER0: address = cREGADDR_RXF0; break;
        case DRV_MCP2515_FILTER1: address = cREGADDR_RXF1; break;
        case DRV_MCP2515_FILTER2: address = cREGADDR_RXF2; break;
        case DRV_MCP2515_FILTER3: address = cREGADDR_RXF3; break;
        case DRV_MCP2515_FILTER4: address = cREGADDR_RXF4; break;
        case DRV_MCP2515_FILTER5: address = cREGADDR_RXF5; break;
        default: address = cREGADDR_RXF0; break;
    }

    // Write buffer
    DRV_MCP2515_BufferWrite(address, d, FILTER_SIZE);
}

void DRV_MCP2515_MaskConfigure(DRV_MCP2515_MASK mask, DRV_MCP2515_MASKOBJ_ID* mObj)
{
#define MASK_SIZE 4

    unsigned char d[MASK_SIZE];
    uint32_t reg = 0;
    unsigned char n=0;

    // RXMnSIDH
    Nop(); Nop();
    reg = mObj->MSID>>3;
    d[n] = reg & 0xff;
    n++;

    // RXMnSIDL
    DRV_MCP2515_REG_TXBnSIDL sidl;
    sidl.byte = 0;
    sidl.SIDL = mObj->MSID & 0x7;
    sidl.IDE = 0; // Not implemented
    reg = (mObj->MEID>>16);
    sidl.EIDHH = reg & 0x3;
    d[n] = sidl.byte;
    n++;

    // RXMnEID8
    reg = (mObj->MEID>>8);
    d[n] = reg & 0xff;
    n++;

    // RXMnEID0
    d[n] = mObj->MEID & 0xff;
    n++;

        // Copy data
    unsigned char address;
    switch(mask) {
        case DRV_MCP2515_MASK0: address = cREGADDR_RXM0; break;
        case DRV_MCP2515_MASK1: address = cREGADDR_RXM1; break;
        default: address = cREGADDR_RXM0; break;
    }

    // Write buffer
    DRV_MCP2515_BufferWrite(address, d, MASK_SIZE);
}

void DRV_MCP2515_ReceiveMessageGet(DRV_MCP2515_RX_CHANNEL channel, DRV_MCP2515_RX_MSGOBJ* rxObj, unsigned char *rxd, unsigned char nBytes)
{
#define RX_BUFFER_SIZE 13

    unsigned char d[RX_BUFFER_SIZE];
    unsigned char n=0;

    // Get address
    unsigned char address;
    switch(channel) {
        case DRV_MCP2515_RX_CH0: address = cREGADDR_RXB0ID; break;
        case DRV_MCP2515_RX_CH1: address = cREGADDR_RXB1ID; break;
        default: address = cREGADDR_RXB0ID; break;
    }

    // Read buffer
    n = nBytes + 5;
    if (n>RX_BUFFER_SIZE) {
        n = RX_BUFFER_SIZE;

    }
    DRV_MCP2515_BufferRead(address, d, n);
    Nop(); Nop();

    // SID
    DRV_MCP2515_REG_RXBnSIDL sidl;
    sidl.byte = d[1];
    rxObj->id.SID = sidl.SIDL;
    rxObj->id.SID += (unsigned int)d[0] <<3;

    // EID
    rxObj->id.EID = d[3];
    rxObj->id.EID += (uint32_t)d[2] <<8;
    rxObj->id.EID += (uint32_t)sidl.EIDHH <<16;

    // Control
    DRV_MCP2515_REG_RXBnDLC dlc;
    dlc.byte = d[4];
    rxObj->ctrl.IDE = sidl.IDE;
    rxObj->ctrl.SRR = sidl.SRR;
    rxObj->ctrl.DLC = dlc.DLC;
    rxObj->ctrl.RTR = dlc.RTR;
    rxObj->ctrl.RB0 = dlc.RB0;
    rxObj->ctrl.RB1 = dlc.RB1;

    // Data
    unsigned int i;
    for (i=0; i<(n-5); i++) {
        rxd[i] = d[i+5];
    }
    Nop(); Nop();

    return;
}

void DRV_MCP2515_ReceiveConfigurationSet(DRV_MCP2515_RX_CONFIG* config)
{
    // Configure RB0
    DRV_MCP2515_REG_RXB0CTRL rxb0Ctrl;
    rxb0Ctrl.byte = 0;
    rxb0Ctrl.RXM = config->RXB0_ReceiveMode;
    rxb0Ctrl.BUKT = config->RXB_RollOver;
    DRV_MCP2515_RegisterWrite(cREGADDR_RXB0CTRL, rxb0Ctrl.byte);

    // Configure RB1
    DRV_MCP2515_REG_RXB1CTRL rxb1Ctrl;
    rxb1Ctrl.byte = 0;
    rxb1Ctrl.RXM = config->RXB1_ReceiveMode;
    DRV_MCP2515_RegisterWrite(cREGADDR_RXB1CTRL, rxb1Ctrl.byte);
}

DRV_MCP2515_MODULE_EVENT DRV_MCP2515_ModuleEventGet()
{
    // Read Interrupt register
    DRV_MCP2515_MODULE_EVENT flags;
    flags = DRV_MCP2515_RegisterRead(cREGADDR_CANINTF);

    return flags;
}

void DRV_MCP2515_ModuleEventEnable(DRV_MCP2515_MODULE_EVENT flags)
{
    // Write Interrupt Enable Register
    DRV_MCP2515_RegisterBitModify(cREGADDR_CANINTE, flags, 0xFF);
}

void DRV_MCP2515_ModuleEventDisable(DRV_MCP2515_MODULE_EVENT flags)
{
    // Write Interrupt Enable Register
    DRV_MCP2515_RegisterBitModify(cREGADDR_CANINTE, flags, 0x00);
}

void DRV_MCP2515_ModuleEventClear(DRV_MCP2515_MODULE_EVENT flags)
{
    // Write Interrupt Flag Register
    DRV_MCP2515_RegisterBitModify(cREGADDR_CANINTF, flags, 0x00);
}

DRV_MCP2515_REG_BUFFER_STATUS DRV_MCP2515_BufferStatusGet()
{
    DRV_MCP2515_REG_BUFFER_STATUS status;

    CAN_CS_LOW();
    SPITransfer(CAN_RD_STATUS);
    status.byte = SPITransfer(0);
    SPITransfer(0);
    CAN_CS_HIGH();

    return status;
}

unsigned char DRV_MCP2515_RegisterRead(unsigned char address)
{
    unsigned char d;

    CAN_CS_LOW();
    SPITransfer(CAN_READ);
    SPITransfer(address);
    d = SPITransfer(0);
    CAN_CS_HIGH();

    return d;
}

void DRV_MCP2515_RegisterWrite(unsigned char address, unsigned char data)
{
    CAN_CS_LOW();
    SPITransfer(CAN_WRITE);
    SPITransfer(address);
    SPITransfer(data);
    CAN_CS_HIGH();
}

void DRV_MCP2515_RegisterBitModify(unsigned char address, unsigned char mask, unsigned char data)
{
    CAN_CS_LOW();
    SPITransfer(CAN_BIT_MODIFY);
    SPITransfer(address);
    SPITransfer(mask);
    SPITransfer(data);
    CAN_CS_HIGH();
}

void DRV_MCP2515_BufferWrite(unsigned char address, unsigned char *data, unsigned char nBytes)
{
    unsigned char n, d;

    CAN_CS_LOW();
    SPITransfer(CAN_WRITE);
    SPITransfer(address);
    for (n=0; n<nBytes; n++) {
        d = data[n];
        Nop();
        Nop();
        SPITransfer(d);
    }
    CAN_CS_HIGH();
}

void DRV_MCP2515_BufferRead(unsigned char address, unsigned char *data, unsigned char nBytes)
{
    unsigned char n, d;

    CAN_CS_LOW();
    SPITransfer(CAN_READ);
    SPITransfer(address);
    for (n=0; n<nBytes; n++) {
        d = SPITransfer(0);
        Nop();
        Nop();
        data[n] = d;
    }
    CAN_CS_HIGH();
}

unsigned char SPITransfer(unsigned char txByte)
{
    // For now use only SPI1
    // Write to SPI
    WriteSPI1(txByte);

    // Wait for data
    while (!DataRdySPI1()) {
        Nop();
    }

    // Return data
    unsigned int rxByte = ReadSPI1();

    return (unsigned char)rxByte;
}
